#define INCL_DOSPROCESS
#define INCL_DOSMEMMGR
#include <os2.h>

#include <sys/builtin.h>
#include <sys/smutex.h>
#include <sys/param.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <errno.h>
#include <sys/errnox.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

static _smutex shm_lock = 0;

static int findkey (key_t key);
static int newseg (key_t key, int shmflg, size_t size);
static int killseg (int id);

static int max_shmid = 0; /* every used id is <= max_shmid */
static struct shmid_ds *shm_segs[SHMMNI];
static struct shmid_ds shm_data[SHMMNI];
static unsigned char defer_rm[SHMMNI];

static unsigned short shm_seq = 0; /* incremented, for recognizing stale ids */

/*
 * Check user, group, other permissions for access
 * to ipc resources. return 0 if allowed
 */
	static int
__SHMx_ipcperms(struct ipc_perm *ipcp, int flag)
{       /* flag will most probably be 0 or S_...UGO from <linux/stat.h> */
        int requested_mode, granted_mode;

        requested_mode = (flag >> 6) | (flag >> 3) | flag;
        granted_mode = ipcp->mode;
        granted_mode >>= 6;
        /* is there some bit set in requested_mode but not in granted_mode? */
        if (requested_mode & ~granted_mode & 0007)
                return -1;
        return 0;
}

/* You must initialize shm! The dll init does it for you, if you are lucky.
 */
	extern void 
__SHMx_shmInit(void)
{
        int id;

        _dPuts("shm_init()\n");
        _smutex_request(&shm_lock);
        for (id = 0; id < SHMMNI; id++)
                shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
        _smutex_release(&shm_lock);
}

	extern void 
__SHMx_shmDone(void)
{
        int id;

        _dPuts("shm_done()\n");
        for (id = 0; id < SHMMNI; id++) {
                _smutex_request(&shm_lock);
                killseg(id);
                shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
                _smutex_release(&shm_lock);
        }
}

static int 
findkey(key_t key)
{
        int id;
        struct shmid_ds *shp;

        for (id = 0; id <= max_shmid; id++) {
                while ((shp = shm_segs[id]) == IPC_NOID) {
                        _smutex_request(&shm_lock);
                        _dPuts("findkey()\n");
                        _smutex_release(&shm_lock);
                }
                if (shp == IPC_UNUSED)
                        continue;
                if (key == shp->shm_perm.key)
                        return id;
        }
        return (int) IPC_UNUSED;
}

/*
 * allocate new shmid_ds. protected by shm_segs[id] = NOID.
 */
static int 
newseg(key_t key, int shmflg, size_t size)
{
        struct shmid_ds *shp;
        int id;
        APIRET rc;

        if (size < SHMMIN)
                return -EINVAL;
        for (id = 0; id < SHMMNI; id++)
                if (shm_segs[id] == IPC_UNUSED) {
                        _smutex_request(&shm_lock);
                        shm_segs[id] = (struct shmid_ds *) IPC_NOID;
                        goto found;
                }
        return -ENOSPC;

found:
        shp = &shm_data[id];

        rc = DosAllocSharedMem(&shp->shm_pages, NULL, size,
        		OBJ_GETTABLE | PAG_COMMIT | PAG_READ | PAG_WRITE);
        if (rc) {
                shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
                _smutex_release(&shm_lock);
                return -ENOMEM;
        } else
          _dPuts("DosAllocSharedMem() successful\n");

        shp->shm_perm.key = key;
        shp->shm_perm.mode = shmflg;
        shp->shm_perm.cuid = shp->shm_perm.uid = 0;
        shp->shm_perm.cgid = shp->shm_perm.gid = 0;
        shp->shm_perm.seq = shm_seq;
        shp->shm_segsz = size;
        shp->shm_cpid = getpid();
        shp->shm_lpid = shp->shm_nattch = 0;
        shp->shm_atime = shp->shm_dtime = 0;
        shp->shm_ctime = time(NULL);

        if (id > max_shmid)
                max_shmid = id;
        shm_segs[id] = shp;
	defer_rm[id] = 0;
        _smutex_release(&shm_lock);
        _dPuts("newseg() successful\n");

        return (unsigned int) shp->shm_perm.seq * SHMMNI + id;
}


	extern int 
shmctl(int shmid, int cmd, struct shmid_ds *buf)
{
        struct shmid_ds tbuf;
        struct shmid_ds *shp;
        struct ipc_perm *ipcp;
        int id;

        if (cmd < 0 || shmid < 0)
                return -EINVAL;
        if (cmd == IPC_SET) {
                return -EINVAL;
        }

        switch (cmd) { /* replace with proc interface ? */
        case IPC_INFO:
        {
                struct shminfo shminfo;
                if (!buf)
                        return -EFAULT;
                shminfo.shmmni = SHMMNI;
                shminfo.shmmax = SHMMAX;
                shminfo.shmmin = SHMMIN;
                shminfo.shmall = SHMALL;
                shminfo.shmseg = SHMSEG;
                memcpy (buf, &shminfo, sizeof(struct shminfo));
                return max_shmid;
        }
        case SHM_INFO:
        {
                return -EINVAL;
        }
        case SHM_STAT:
                if (!buf)
                        return -EFAULT;
                if (shmid > max_shmid)
                        return -EINVAL;
                shp = shm_segs[shmid];
                if (shp == IPC_UNUSED || shp == IPC_NOID)
                        return -EINVAL;
                if (__SHMx_ipcperms(&shp->shm_perm, S_IRUGO))
                        return -EACCES;
                id = (unsigned int) shp->shm_perm.seq * SHMMNI + shmid;
                tbuf.shm_perm   = shp->shm_perm;
                tbuf.shm_segsz  = shp->shm_segsz;
                tbuf.shm_atime  = shp->shm_atime;
                tbuf.shm_dtime  = shp->shm_dtime;
                tbuf.shm_ctime  = shp->shm_ctime;
                tbuf.shm_cpid   = shp->shm_cpid;
                tbuf.shm_lpid   = shp->shm_lpid;
                tbuf.shm_nattch = shp->shm_nattch;
                memcpy (buf, &tbuf, sizeof(*buf));
                return id;
        }

        shp = shm_segs[id = (unsigned int) shmid % SHMMNI];
        if (shp == IPC_UNUSED || shp == IPC_NOID)
                return -EINVAL;
        if (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)
                return -EIDRM;
        ipcp = &shp->shm_perm;

        switch (cmd) {
        case SHM_UNLOCK:
                return -EINVAL;
        case SHM_LOCK:
                return -EINVAL;
        case IPC_STAT:
                if (__SHMx_ipcperms(ipcp, S_IRUGO))
                        return -EACCES;
                if (!buf)
                        return -EFAULT;
                tbuf.shm_perm   = shp->shm_perm;
                tbuf.shm_segsz  = shp->shm_segsz;
                tbuf.shm_atime  = shp->shm_atime;
                tbuf.shm_dtime  = shp->shm_dtime;
                tbuf.shm_ctime  = shp->shm_ctime;
                tbuf.shm_cpid   = shp->shm_cpid;
                tbuf.shm_lpid   = shp->shm_lpid;
                tbuf.shm_nattch = shp->shm_nattch;
                memcpy (buf, &tbuf, sizeof(*buf));
                break;
        case IPC_SET:
                return -EPERM;
        case IPC_RMID:
                if (shp->shm_nattch <= 0) {
                  _smutex_request(&shm_lock);
                  killseg (id);
                  _smutex_release(&shm_lock);
                }
		else
		  defer_rm[id] = 1;
                break;
        default:
                return -EINVAL;
        }
        return 0;
}

	extern int 
shmget(key_t key, int size, int shmflg)
{
        struct shmid_ds *shp;
        int id = 0;

        if (size < 0 || size > SHMMAX)
                return -EINVAL;
        if (key == IPC_PRIVATE)
                return newseg(key, shmflg, (size_t)size);
        if ((id = findkey (key)) == -1) {
                if (!(shmflg & IPC_CREAT))
                        return -ENOENT;
                return newseg(key, shmflg, (size_t)size);
        }
        if ((shmflg & IPC_CREAT) && (shmflg & IPC_EXCL))
                return -EEXIST;
        shp = shm_segs[id];
        if (size > shp->shm_segsz)
                return -EINVAL;
        if (__SHMx_ipcperms(&shp->shm_perm, shmflg))
                return -EACCES;
        _dPuts("shmget() successful");
        return (unsigned int) shp->shm_perm.seq * SHMMNI + id;
}


/*
 * Here pages and shmid_ds are freed.
 */
static int 
killseg(int id)
{
        struct shmid_ds *shp;
        int i, numpages;
        APIRET rc;

        shp = shm_segs[id];
        if (shp == IPC_NOID || shp == IPC_UNUSED) {
                return -EFAULT;
        }
        shp->shm_perm.seq++;     /* for shmat */
        shm_seq = (shm_seq+1) % ((unsigned)(1<<30)/SHMMNI); /* increment, but avoid overflow */
        shm_segs[id] = (struct shmid_ds *) IPC_UNUSED;
        if (id == max_shmid)
                while (max_shmid && (shm_segs[--max_shmid] == IPC_UNUSED));
        if (!shp->shm_pages) {
                return -EFAULT;
        }
	defer_rm[id] = 0;

#if 0
/* DosFreeMem() decrements a reference count,
   so it should be called in shmdt() */
        rc = DosFreeMem(shp->shm_pages);
        if (rc)
          _dPuts("DosFreeMem() failed\n");
        else
          _dPuts("DosFreeMem() successful\n");
#endif

        _dPuts("killseg() successful");
        return IPC_PRIVATE;
}

/*
 * Fix shmaddr, allocate descriptor, map shm, add attach descriptor to lists.
 */
extern void *
shmat(int shmid, const void *shmaddr, int shmflg)
{
        struct shmid_ds *shp;
        int err;
        unsigned int id;
        APIRET rc;

        if (shmid < 0) {
                return (void *) -EINVAL;
        }

        shp = shm_segs[id = (unsigned int) shmid % SHMMNI];
        if (shp == IPC_UNUSED || shp == IPC_NOID) {
                return (void *) -EINVAL;
        }

        if (shmaddr) {
                return (void *) -EINVAL;
        }

_dPuts("shmat() 1 ...\n");

        if (__SHMx_ipcperms(&shp->shm_perm, shmflg & SHM_RDONLY ? S_IRUGO : S_IRUGO|S_IWUGO))
                return (void *) -EACCES;
        if (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)
                return (void *) -EIDRM;


        if ((shp != shm_segs[id]) || (shp->shm_perm.seq != (unsigned int) shmid / SHMMNI)) {
                return (void *) -EIDRM;
        }

_dPuts("shmat() 2 ...\n");

        shp->shm_nattch++;            /* prevent destruction */

        shp->shm_lpid = getpid();
        shp->shm_atime = time(NULL);

        rc = DosGetSharedMem(shp->shm_pages, PAG_READ | PAG_WRITE );
        if (rc) {
                _dPuts("DosGetSharedMem() failed\n");
                return (void *) -EFAULT;
        } else
          _dPuts("DosGetSharedMem() successful\n");

        return shp->shm_pages;
}

/*
 * detach segment
 */
extern int 
shmdt(const void *shmaddr)
{

        int id;
        struct shmid_ds *shp;

        for (id = 0; id <= max_shmid; id++) {
                while ((shp = shm_segs[id]) == IPC_NOID) {
                        _smutex_request(&shm_lock);
                        _smutex_release(&shm_lock);
                }
                if (shp == IPC_UNUSED)
                        continue;
                if (((caddr_t)shp->shm_pages <= (caddr_t)shmaddr)
                && ((caddr_t)shmaddr <= ((caddr_t)shp->shm_pages
                									+ shp->shm_segsz))) {
                        if (shp->shm_nattch) {
			  shp->shm_nattch--;
			/* DosFreeMem() decrements a reference count,
			   so it should be called here */
		          DosFreeMem(shp->shm_pages);
			}
			if (shp->shm_nattch == 0 && defer_rm[id]) killseg(id);
                        _dPuts("shmdt() successful\n");
                        return IPC_PRIVATE;
                }
        }
        return (int) IPC_UNUSED;
}

/* Display information about active shared memory segments */
	extern void
_SHMx_showActiveSeg(void)
{
        int id;
        struct shmid_ds *shp;

	printf("active shared memory segments\n");
	printf("=============================\n");
        for (id = 0; id <= max_shmid; id++) {
                while ((shp = shm_segs[id]) == IPC_NOID) {
                        _smutex_request(&shm_lock);
                        _smutex_release(&shm_lock);
                }
                if (shp != IPC_UNUSED || defer_rm[id]) {
                        _smutex_request(&shm_lock);
			printf("id = %d, key = %ld, nattch = %d\n",
				id, shp->shm_perm.key, shp->shm_nattch);
                        _smutex_release(&shm_lock);
                }
        }
}

