static char ID[]="$Id: main.c,v 3.15 1996/03/10 12:35:18 rl Rel $";
/**********************************************************
* kaleido
*
*	Kaleidoscopic construction of uniform polyhedra
*	Copyright (C) 1991-1996 Dr. Zvi Har'El <rl@math.technion.ac.il>
*
*	This program is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation; either version 2 of the License, or
*	( at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of 
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*	Author:
*	Dr. Zvi Har'El,
*	Deptartment of Mathematics,
*	Technion, Israel Institue of Technology,
*	Haifa 32000, Israel.
*	E-Mail: rl@math.technion.ac.il 
**********************************************************/
#include "kaleido.h" 
static char *message[]  = {
	"Kaleidoscopic Construction of Uniform Polyhedra, $Revision: 3.15 $\n",
	" Copyright (C) 1991-1996 Dr. Zvi Har'El <rl@math.technion.ac.il>\n",
	"Usage: ",
	"", /* stick program name here */
	" [options] [symbols], where the options are:\n",
	"-s: display polyhedra in the Standard list.\n",
	"-i: display polyhedra in an Input list (one per line).\n",
	"-l: List polyhedron names, symbols and reference figures.\n",
	"-x: print successive approXimations on stderr.\n",
	"-v: print Vertex and face coordinates.\n",
	"-d digits: number of significant Digits.\n",
	"-a degrees: rotation axis Azimouth.\n",
	"-e degrees: rotation axis Elevation.\n",
	"-f degrees: angle of rotation until Freezing.\n",
	"-[pP] prefix: prefix for Pic snapshots.\n",
#ifndef NOSCOPE
	"-g: display polyhedron on a Graphics monitor\n",
	"    (hit Enter for next, Backspace for previous, Tab for dual,\n",
	"    S for a snapshot, F to freeze, Esc to exit).\n",
	"-n: display Numerical data in addition to graphical.\n",
	"-L: use a Light background.\n",
	"-[RGB]: display colors: Red, Green, Blue, or combinations.\n",
	" -m mode: video Mode.\n",
	"-r edges: drawing Rate: edges per revolution.\n",
#endif
	"Unless the -s or -i options are selected, the command line may contain\n ",
	"polyhedron symbols: Wythoff symbols (3 fractions and a |) \n",
	"or indices (# and an integer in the range 1 through ",
	0
};

usage()
{
	char **p;
	for (p = message; *p; p++) fputs (*p, stderr);
	fprintf (stderr, " %d).\n ", last_uniform);
	exit (1);
}

warning ()
{
	if (!error.message) {
		if (errno > 0 && errno < sys_nerr) error.message =
sys_errlist[errno];
		errno = 0;
	}
	fprintf (stderr, "Error: %s [%s:%d]\n", error.message,
		error.file, error.line);
}

errexit ()
{
#ifndef NOSCOPE
	if (gcont) gend();
#endif
	warning ();
	exit (1);
} 

main (argc, argv)
int argc;
char **argv;
{
	static digits = 6;
	static double freeze;
	static double azimuth = AZ;
	static double elevation = EL;
	static char *prefix;
	static char *Prefix;
	static mode = -1;
	static rate = 25200;
	static more_lines = 23;
	static index, input, just_list, graphics, numeric, color;
	static need_coordinates, need_approx, need_edgelist;
	char *s;
/*
* Process command line options
*/
	message[3] = *argv;
	if (argc == 1) usage();
	while (--argc)
		if (*(s = *++argv) == '-') {
			while (*++s) switch (*s) {
				char *argp;
			case 'd':
				digits = atoi (VAL);
				break;
			case 'f':
				need_edgelist = 1;
				freeze = atof (VAL) / DEG;
				break;
			case 'a':
				azimuth = atof (VAL) / DEG;
				break;
			case 'e':
				elevation = atof (VAL) / DEG;
				break;
			case 'p':
				prefix = VAL;
				break;
			case 'P':
				Prefix = VAL;
				break;
			case 'l':
				just_list = 1;
				break;
			case 's':
				if (!input) input = 1;
				break;
			case 'i':
				input = -1;
				break;
			case 'v':
				need_coordinates = 1;
				numeric = 1;
				break;
			case 'x':
				need_approx = 1;
				break;
#ifndef NOSCOPE
			case 'g':
				need_edgelist = 1;
				graphics = 1;
				break;
			case 'n':
				numeric = 1;
				break;
			case 'R':
				color |= 1;
				break;
			case 'G':
				color |= 2;
				break;
			case 'B':
				color |= 4;
				break;
			case 'L':
				color |= 8;
				break;
			case 'm':
				mode = atoi (VAL);
				break;
			case 'r':
				rate = atoi (VAL);
				break;
#endif
			default:
				fprintf (stderr, " illegal option: -%c\n", *s);
				usage ();
			}
		} else break; /* rest of arguments are symbols */
/*
* Process enviroment variables
*/
	if (s = getenv ("LINES"))
		more_lines = atoi (s) - 1;
/*
*Set defaults
*/
	if (need_edgelist) {
		if (!prefix && !Prefix) {
			static char default_prefix[9] = "pic";
			sprintf (default_prefix + 3, "%d", getpid());
			Prefix = prefix = default_prefix;
		} else if (!prefix) prefix = Prefix;
		else if (!Prefix) Prefix = prefix;
	}
	if (!graphics) numeric = 1;
/*
* Initialize continuous graphics
*/
#ifndef NOSCOPE
	if (graphics) {
		if (!gcont && !numeric && !(input < 0 &&
			(isatty (fileno (stdin)) || isatty (fileno (stdout))))) gcont = 1;
		if (gcont &&  !ginit(mode, color)) errexit ();
	} else gcont = 0;
#endif
/*
* Main loop
*/
	for (;;) {
		char *sym;
		Polyhedron *P;
/*
* Get next polyhedron
*/
		if (!(sym = nextsym(input, index, &argc, &argv))) break;
/*
* Solve polyhedron
*/
		if (!(P = kaleido (sym, need_coordinates,
			need_edgelist, need_approx, just_list))) {
			if (input >= 0 || !isatty(fileno(stdin)) ||
				!isatty(fileno(stdout))) errexit ();
			warning ();
			continue;
		}
/*
* Print data
*/
		if (numeric && !printit (P, need_coordinates, just_list, digits, more_lines))
			errexit();
/*
* Show pictures of polyhedron and its dual.
*/
		if (graphics) {
#ifndef NOSCOPE
			int c;
			if (!gcont &&  !ginit(mode, color)) errexit ();
			c=gshow (P, rate, azimuth, elevation, freeze, prefix, Prefix);
			if (!gcont) gend ();
			if (!c) break;
			if (input > 0) index += c;
#endif
/*
* Save picture on file
*/
		} else {
			if (need_edgelist && !savepic (P, azimuth, elevation,
				freeze, prefix, Prefix)) errexit();
			if (input > 0) index++;
		}
		polyfree (P);
	} 
/*
* Normal termination
*/ 
#ifndef NOSCOPE
	if (gcont) gend();
#endif
	exit (0);
}

char *
nextsym (input, index, pargc, pargv)
int input, index, *pargc;
char ***pargv;
/*
* Returns the next input symbol.
*/
{
	if (!input){ /* get symbol from command line */
		if (!(*pargc)--) return 0;
		return *(*pargv)++;
	} else if (input > 0) { /* get symbol from standard list */
		static char sym[4];
		if (index < 0 || index >= last_uniform) return 0;
		sprintf (sym, " #%d ", index + 1);
		return sym;
	} else { /* get symbol from standard input */
		int c, len = 0;
		static char *sym;
		if (!sym) Malloc (sym,1, char)
		if (isatty(fileno(stdout)) && isatty(fileno(stdin)))
			fputs (" Enter symbol: ",stdout);
		if ((c = getchar ()) == EOF) {
			if (isatty(fileno(stdout)) && isatty(fileno(stdin)))
			fputs ("EOF\n", stdout);
			return 0;
		}
		ungetc (c, stdin);
		while ((c = getchar ()) != EOF && c != '\n') {
			sym[len++] = c;
			Realloc (sym, len+1, char)
		}
		sym[len] = 0;
		return sym;
	}
}

printit (P, need_coordinates,just_list, digits, more_lines)
Polyhedron *P;
int need_coordinates, just_list, digits, more_lines;
/*
* Print polyhedron data
*/
{
	int j, i;
	double cosa;
	static char *group[] = {"di", "tetra", "octa", "icosa"};
	static char *alias[] = {"D", "A4", "S4", "A5"};
/*
* Print polyhedron name, Wythoff symbol, and reference figures.
*/
	if (P->index != -1) printf ("%d) ", P->index + 1);
	printf ("%s %s", P->name, P->polyform);
	if (P->index != -1 && uniform[P->index].Coxeter)
		printf (" [%d,%d]",uniform[P->index].Coxeter,
uniform[P->index].Wenninger);
	putchar ('\n');
	if (just_list) return 1;
/*
* Print combinatorial description.
*/
	printf ("\tdual: %s\n\t%s, %shedral group %s", P->dual_name,
		P->config, group[P->K-2], alias[P->K-2]);
	if (P->K == 2) printf ("%d", P->g / 4);
	printf (", chi=%d", P->chi);
	if (P->D) printf (", D=%d", P->D);
	else if (P->onesided) printf (", one-sided");
	else printf (", two-sided");
	printf ("\n\tV=%d, E=%d, F=%d=", P->V, P->E, P->F);
	for (j = 0; j < P->N; j++) {
		if (j) putchar ('+');
		printf ("%d{", P->Fi[j]);
		printfrac (P->n[j]);
		putchar ('}');
	}
/*
* Print solution.
*/
	if (P->index == -1 && P->K == 2) {
		char *s;
		Sprintfrac (s, P->gon)
		i = strlen (s) + 2;
		if (i < 6) i = 6;
		free (s);
	} else i = 6;
	printf ("\n%*s%6s%*s%*s%*s%*s%*s%*s%*s%*s\n",
		i, "", "alpha", digits + 3, "gamma", digits + 1, "a",
		digits + 1, "b", digits + 1, "c", digits + 3, "rho/R",
		digits + 3, "r/rho", digits + 3, "l/rho", digits + 3, "h/r");
	cosa = cos (M_PI / P->n[0]) / sin (P->gamma[0]);
	for (j = 0; j < P->N; j++) {
		double cosc = cos (P->gamma[j]) / sin (M_PI / P->n[j]);
		char *s, *t;
		Sprintfrac (s, P->n[j])
		Malloc (t, strlen (s) + 3, char)
		sprintf (t, "{%s}", s);
		free (s);
		printf ("%*s%6.1f%*.*f%*.*f%*.*f%*.*f%*.*f%*.*f",
			i, t, 180. / P->n[j],
			digits + 3, digits - 2, DEG * P->gamma[j],
			digits + 1, digits - 4, DEG * acos (cosa),
			digits + 1, digits - 4, DEG * acos (cosa * cosc),
			digits + 1, digits - 4, DEG * acos (cosc),
			digits + 3, digits, cosa,
			digits + 3, digits, cosc);
		if (log10 (fabs (cosa)) < -digits) printf ("%*s", digits + 3,
		"infinity");
		else printf ("%*.*f", digits + 3, digits, sqrt (1 - cosa * cosa)
/ cosa);
		if (log10 (fabs (cosc)) < -digits) printf ("%*s", digits + 3,
"infinity");
		else printf ("%*.*f", digits + 3, digits, sqrt (1 - cosc * cosc)
/ cosc);
		putchar ('\n');
		free (t);
	}
	putchar ('\n');
	more ();
	if (!need_coordinates) return 1;
/*
* Print vertices
*/ 
	printf ("vertices:\n");
	for (i = 0; i < P->V; i++) {
		printf ("v%-3d (%*.*f,%*.*f,%*.*f)",
			i+1 ,digits + 3, digits, P->v[i].x, digits + 3,
			digits, P->v[i].y, digits + 3, digits, P->v[i].z);
		for (j = 0; j < P->M; j++) printf (" v%-3d",P->adj[j][i] + 1);
		printf ("\n%*s", 3 * digits + 20, "");
		for (j = 0; j < P->M; j++) printf (" f%-3d",P->incid[j][i] + 1);
		putchar ('\n');
		if (!((i + 1) % (more_lines / 2))) more();
	}
	if (P->V % (more_lines / 2)) more ();
/*
* Print faces.
*/
	printf ("faces (RHS=%*.*f ", digits + 2, digits, P->minr);
	printf ("):\n");
	for (i = 0; i < P->F; i++){
		printf ("f%-3d (%*.*f,%*.*f,%*.*f) {",
			i+1 ,digits + 3, digits, P->f[i].x, digits + 3,
			digits, P->f[i].y, digits + 3, digits, P->f[i].z);
		printfrac (P->n[P->ftype[i]]);
		putchar ('}');
		if (P->hemi && !P->ftype[i]) putchar ('*');
		putchar ('\n');
		if (!((i + 1) % more_lines)) more();
	}
	if (P->F % more_lines) more ();
	putchar ('\n');
	return 1;
}

more () 
{
	int c;
	if (isatty(fileno(stdout)) && isatty(fileno(stdin))) {
		fputs ("Enter for more...", stdout);
		while ((c = getchar ()) != EOF && c != '\n') ;
	}
}

savepic (P, azimuth, elevation, freeze, prefix, Prefix)
Polyhedron *P;
double azimuth, elevation, freeze;
char *prefix, *Prefix;
{
	char *fn;
	rotframe (azimuth, elevation, freeze);
	fn = picfile (P->v, P->V, P->e, P->E, ( int *) 0,
		P->index, "", P->config, P->polyform, prefix, 1);
	if (!fn) Err (0)
	fprintf (stderr, "%s: saved on %s\n", P->name, fn);
	free (fn);
	fn = picfile (P->f, P->F, P->dual_e, P->E, P->anti,
		P->index, "*", P->config, P->polyform, Prefix, 1);
	if (!fn) Err (0)
	fprintf (stderr, "%s: saved on %s\n", P->dual_name, fn);
	free (fn);
	return 1;
}

char *
picfile (v, V, e, E, anti, index, star, subtitle, subsub, prefix, need_rotation)
Vector *v;
int V, E, need_rotation, **e, *anti, index;
char *star, *subtitle, *subsub, *prefix;
/*
* Generate a pic file
*/
{
	int i;
	FILE *fp = 0;
	Vector temp, *new = 0;
	int *s, *t, *u;
	char *fn;

	Malloc (fn, strlen (prefix) + 5, char)
	for (i = 0; i < 1000; i++) {
		sprintf (fn, "%s.%03d", prefix, i);
		if (access (fn, 0)) {/* file doesn't exist */
			fp = fopen (fn, "w");
			break;
		}
	}
	if (i == 1000) Err ("too many files")
	if (!fp) Err (0)
	if (!need_rotation) new = v;
	else {
		Malloc (new, V, Vector)
		rotarray (new, v, V);
	}
	fprintf (fp, ".PS\n");
	fprintf (fp, ".ps 8\nbox ht 2.5 wid 2.5 at 0,0\n");
	fprintf (fp, "\"%s \" at last box.se rjust above\n", subtitle);
	fprintf (fp, "\" %s\" at last box.sw ljust above\n", subsub);
	if (index != -1) fprintf (fp, "\"Fig. %d%s\" at last box.s below\n",
			index + 1, star);
	s = e[0];
	t = e[1];
	u = anti;
	for (i = E; !ferror (fp) && i--; ) {
		int j=* s++, k = *t++;
		if (!anti) {
			pic (fp, new[j], new[k], j, k);
		} else if (*u++) {
			temp = scale (.5, new[j]);
			pic (fp, temp, diff (temp, new[k]), j, k);
		} else {
			temp = scale (.5, new[j]);
			pic (fp, temp, sum (temp, new[k]), j, k);
		}
	} fprintf (fp, ".PE\n");
	if (ferror (fp)) Err (0);
	fclose (fp);
	if (need_rotation) Free (new)
	return fn;
}

pic (fp, a, b, j, k) 
FILE *fp;
Vector a, b;
int j, k;
/* generate pic command to draw a segment using variable pointsize */
{
	int i;
	double aa, bb;
	if (a.z < b.z) {/*swap endpoints to get increasing point size*/
		Vector temp;
		temp = a;
		a = b;
		b = temp;
		i = j;
		j = k;
		k = i;
	}
	aa = 8 * (1 - a.z);
	bb = 8 * (1 - b.z);
	fprintf (fp, "move to %g,%g #v%d\n", a.x, a.y, j);
	for (i = (int) aa+1; i < (int) bb + 1; i++) {
		double f;
		f = ((double) i - aa) / (bb - aa);
		fprintf (fp, ".ps %d\nline to %g,%g\n", i,
			a.x + (b.x - a.x) * f,
			a.y + (b.y - a.y) * f);
	}
	fprintf (fp, ".ps %d\nline to %g,%g #v%d\n", i, b.x, b.y, k);
}
